/****************************************************************************
 *
 * mpi-vecvec.c - Test MPI derived datatype
 *
 * Copyright 2021 Moreno Marzolla <moreno.marzolla(at)unibo.it>
 * Modified by Copyright 2025  Roberto Giorgi <giorgi@acm.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * --------------------------------------------------------------------------
 *
 * Compile with:
 * mpicc -std=c99 -Wall -Wpedantic mpi-vecvec.c -o mpi-vecvec
 *
 * Run with:
 * mpirun -n 2 ./mpi-vecvec
 *
 ****************************************************************************/

#include <stdio.h>
#include "mpi.h"

int main( int argc, char *argv[] )
{
    int my_rank, comm_sz;
    int count, blocklen, stride;
    MPI_Datatype vec, vecvec;
    float v[36];
    int i, error = 0;

    MPI_Init(&argc, &argv);
    MPI_Comm_rank(MPI_COMM_WORLD, &my_rank);
    MPI_Comm_size(MPI_COMM_WORLD, &comm_sz);

    count = 2; blocklen = 2; stride = 3;
    MPI_Type_vector(count, blocklen, stride, MPI_FLOAT, &vec);
    MPI_Type_commit(&vec);
    count = 2; blocklen = 1; stride = 3;
    MPI_Type_vector(count, blocklen, stride, vec, &vecvec);
    MPI_Type_commit(&vecvec);
    MPI_Aint lb, extent;
    MPI_Type_get_extent(vecvec, &lb, &extent);
    printf("lb=%ld extent=%ld\n", lb, extent);
    for (i=0; i<36; i++) {
        if (my_rank == 0)
            v[i] = i;
        else if (my_rank == 1)
            v[i] = -1;
        else
            /* Note: All ranks other than 0 will call MPI_Recv, but only one 
               message is sent to rank 1. This will block for ranks ≥2. */
            error = 1;
    }

    if (! error) {
        /* v[] nel processo 0: [0, 1, 2, ... 35];

           v[] negli altri processi: [-1, -1, ... -1]; */
        
        if (my_rank == 0) {
            MPI_Send(v,             /* sendbuf      */
                     1,             /* count        */
                     vecvec,        /* datatype     */
                     1,             /* dest         */
                     0,             /* tag          */
                     MPI_COMM_WORLD);
        } else {
            MPI_Recv(v,             /* recvbuf      */
                     1,             /* recvcount    */
                     vecvec,        /* recvtype     */
                     0,             /* source       */
                     MPI_ANY_TAG,   /* tag          */
                     MPI_COMM_WORLD,
                     MPI_STATUS_IGNORE);
            for (i=0; i<36; i++) {
                printf("[%2d]=%2g ", i, v[i]);
                if (i % 6 == 5) printf("\n");
            }
            printf("\n");
        }
    } else {
        printf("Only 2 ranks are suported.\n");
    }
    MPI_Type_free(&vecvec);
    MPI_Type_free(&vec);
    MPI_Finalize();
    return 0;
}
